<?php
/**
 * autoresult.php
 * Final smart auto-updater + responsive UI (with formatted AM/PM times)
 */

error_reporting(E_ALL & ~E_NOTICE);
ini_set('display_errors', 0);
date_default_timezone_set("Asia/Kolkata");

// --- DB connect ---
$mysqli = new mysqli("localhost:3306", "matkabazarfast_matkauser", "Ashok@2025", "matkabazarfast_matkadb");
if ($mysqli->connect_error) {
    die("<p style='color:red'>DB connection failed: " . htmlspecialchars($mysqli->connect_error) . "</p>");
}

// ✅ Format time in 12-hour style (AM/PM)
function formatTime($timeStr) {
    $t = trim($timeStr);
    if ($t == '') return '-';
    $ts = strtotime($t);
    return $ts ? date('h:i A', $ts) : $t;
}

// --- helper: normalize time ---
function normalizeTime($t) {
    $t = trim((string)$t);
    if ($t === '') return null;
    $t = str_replace(['.', '-'], ':', $t);
    $t = preg_replace('/\s+/', ' ', $t);
    if (preg_match('/\b(am|pm)\b/i', $t)) {
        $ts = strtotime($t);
        return $ts ? date('H:i:s', $ts) : null;
    }
    if (preg_match('/^\d{1,2}:\d{2}$/', $t)) {
        $ts = strtotime($t);
        return $ts ? date('H:i:s', $ts) : null;
    }
    $ts = strtotime($t);
    return $ts ? date('H:i:s', $ts) : null;
}

// --- fetch dpboss homepage ---
$sourceUrl = "https://dpboss.boston/";
$html = @file_get_contents($sourceUrl);
if (!$html) {
    die("<p style='color:red'>ERROR: Could not fetch dpboss homepage: $sourceUrl</p>");
}

libxml_use_internal_errors(true);
$dom = new DOMDocument();
$dom->loadHTML($html);
libxml_clear_errors();
$xpath = new DOMXPath($dom);

// --- parse dpboss market blocks ---
$blocks = $xpath->query("//div[h4 and span]");
$dpResults = [];
foreach ($blocks as $div) {
    $h4 = $xpath->query(".//h4", $div)->item(0);
    $span = $xpath->query(".//span", $div)->item(0);
    if (!$h4 || !$span) continue;
    $marketText = trim($h4->textContent);
    $resultText = trim($span->textContent);
    if ($marketText === '' || $resultText === '') continue;
    $norm = mb_strtoupper(preg_replace('/\s+/', ' ', $marketText));
    $dpResults[$norm] = preg_replace('/\s+/', '', $resultText);
}

// --- load markets from DB ---
$games = [];
$res = $mysqli->query("SELECT `game`, `open`, `close`, `days` FROM `game`");
if ($res) {
    while ($row = $res->fetch_assoc()) {
        $name = trim($row['game']);
        if ($name === '') continue;
        $norm = mb_strtoupper(preg_replace('/\s+/', ' ', $name));
        $games[$norm] = [
            'game' => $row['game'],
            'open' => $row['open'],
            'close'=> $row['close'],
            'days' => (int)$row['days']
        ];
    }
    $res->free();
}

// --- helper: active day logic ---
function isActiveToday($days, $dow) {
    if ($days >= 7) return true;
    if ($days == 6 && $dow <= 6) return true;
    if ($days == 5 && $dow <= 5) return true;
    return false;
}

// --- runtime vars ---
$today = date('Y-m-d');
$now = date('H:i:s');
$todayDow = date('N');

$counts = ['inserted'=>0, 'updated'=>0, 'skipped'=>0, 'nochange'=>0, 'waiting'=>0];
$rowsForDisplay = [];

// --- process each market ---
foreach ($games as $norm => $cfg) {
    $marketName = $cfg['game'];
    $openTime = normalizeTime($cfg['open']);
    $closeTime = normalizeTime($cfg['close']);
    $days = (int)$cfg['days'];

    // ✅ Display formatted times in AM/PM
    $openFormatted = formatTime($cfg['open']);
    $closeFormatted = formatTime($cfg['close']);
    $timeLabel = $openFormatted . " → " . $closeFormatted;

    if (!isset($dpResults[$norm])) continue;

    $dpResult = $dpResults[$norm];
    $tableName = "game_" . str_replace(' ', '_', mb_strtoupper($marketName));
    $tableCheck = $mysqli->query("SHOW TABLES LIKE '".$mysqli->real_escape_string($tableName)."'");
    if (!$tableCheck || $tableCheck->num_rows == 0) {
        $rowsForDisplay[] = [$marketName, $timeLabel, "red", "Skipped (no table)", "-"];
        $counts['skipped']++;
        continue;
    }

    $stmt = $mysqli->prepare("SELECT result FROM `$tableName` WHERE result_date = ? LIMIT 1");
    if (!$stmt) {
        $rowsForDisplay[] = [$marketName, $timeLabel, "red", "Skipped (db error)", "-"];
        $counts['skipped']++;
        continue;
    }
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $res = $stmt->get_result();
    $todayRow = $res->fetch_assoc();
    $stmt->close();

    if ($todayRow && !empty($todayRow['result'])) {
        if ($todayRow['result'] === $dpResult) {
            $rowsForDisplay[] = [$marketName, $timeLabel, "black", "Already inserted", htmlspecialchars($todayRow['result'])];
            $counts['nochange']++;
            continue;
        }
        if ($now < $openTime) {
            $rowsForDisplay[] = [$marketName, $timeLabel, "blue", "Waiting for open ($openFormatted)", htmlspecialchars($todayRow['result'])];
            $counts['waiting']++;
            continue;
        }
        $upd = $mysqli->prepare("UPDATE `$tableName` SET result = ?, date = NOW() WHERE result_date = ?");
        if ($upd) {
            $upd->bind_param("ss", $dpResult, $today);
            $ok = $upd->execute();
            $upd->close();
            if ($ok) {
                $rowsForDisplay[] = [$marketName, $timeLabel, "yellow", "Updated", htmlspecialchars($dpResult)];
                $counts['updated']++;
            } else {
                $rowsForDisplay[] = [$marketName, $timeLabel, "red", "Skipped (update failed)", "-"];
                $counts['skipped']++;
            }
        } else {
            $rowsForDisplay[] = [$marketName, $timeLabel, "red", "Skipped (update prepare failed)", "-"];
            $counts['skipped']++;
        }
        continue;
    }

    if ($now < $openTime) {
        $rowsForDisplay[] = [$marketName, $timeLabel, "blue", "Waiting for open ($openFormatted)", "-"];
        $counts['waiting']++;
        continue;
    }

    if (!isActiveToday($days, $todayDow)) {
        $prevQ = $mysqli->query("SELECT result_date, result FROM `$tableName` WHERE result <> '' ORDER BY result_date DESC LIMIT 1");
        $prev = $prevQ ? $prevQ->fetch_assoc() : null;
        if ($prev && $prev['result'] === $dpResult) {
            $rowsForDisplay[] = [$marketName, $timeLabel, "red", "Carry-forward (inactive day — ".$prev['result_date'].")", htmlspecialchars($dpResult)];
            $counts['skipped']++;
            continue;
        } else {
            $rowsForDisplay[] = [$marketName, $timeLabel, "red", "Skipped (inactive day)", "-"];
            $counts['skipped']++;
            continue;
        }
    }

    $ins = $mysqli->prepare("INSERT INTO `$tableName` (name, result, result_date, date, status) VALUES (?, ?, ?, NOW(), '0')");
    if ($ins) {
        $ins->bind_param("sss", $marketName, $dpResult, $today);
        $ok = $ins->execute();
        $ins->close();
        if ($ok) {
            $rowsForDisplay[] = [$marketName, $timeLabel, "green", "Inserted", htmlspecialchars($dpResult)];
            $counts['inserted']++;
        } else {
            $rowsForDisplay[] = [$marketName, $timeLabel, "red", "Skipped (insert failed)", "-"];
            $counts['skipped']++;
        }
    } else {
        $rowsForDisplay[] = [$marketName, $timeLabel, "red", "Skipped (insert prepare failed)", "-"];
        $counts['skipped']++;
    }
}

// --- build HTML ---
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>AutoResult Run — Report</title>
<style>
body { font-family: "Segoe UI", Roboto, Arial; background:#f4f6f8; margin:0; padding:20px; color:#222; }
.box { max-width:1100px; margin:20px auto; background:#fff; border-radius:10px; box-shadow:0 6px 20px rgba(0,0,0,0.08); padding:20px; }
h1 { margin:0 0 12px 0; font-size:20px; text-align:center; color:#111; }
.summary { display:flex; flex-wrap:wrap; gap:12px; justify-content:center; margin-bottom:16px; }
.card { padding:10px 14px; border-radius:8px; min-width:120px; text-align:center; font-weight:700; }
.card.green { background:#e6f5ea; color:#1b7a3b; }
.card.yellow { background:#fff7e0; color:#7a5b00; }
.card.red { background:#fdecea; color:#8a1f1f; }
.card.black { background:#efefef; color:#2a2a2a; }
.card.blue { background:#e7f0ff; color:#0b56b6; }
table { width:100%; border-collapse:collapse; margin-top:10px; }
th, td { padding:10px 12px; border-bottom:1px solid #eee; text-align:left; font-size:14px; }
th { background:#fafafa; font-weight:800; color:#333; position:sticky; top:0; }
.status { padding:6px 10px; border-radius:6px; font-weight:700; display:inline-block; }
.status.green { background:#28a745; color:#fff; }
.status.yellow { background:#ffc107; color:#000; }
.status.red { background:#dc3545; color:#fff; }
.status.black { background:#343a40; color:#fff; }
.status.blue { background:#007bff; color:#fff; }
@media (max-width:700px){
.summary { flex-direction:column; align-items:stretch; }
th, td { font-size:13px; padding:8px 10px; }
}
</style>
</head>
<body>
<div class="box">
<h1>🏁 AUTORESULT RUN — <?= date('Y-m-d H:i:s') ?></h1>
<div class="summary">
<div class="card green">Inserted: <?= $counts['inserted'] ?></div>
<div class="card yellow">Updated: <?= $counts['updated'] ?></div>
<div class="card red">Skipped: <?= $counts['skipped'] ?></div>
<div class="card black">No change: <?= $counts['nochange'] ?></div>
<div class="card blue">Waiting: <?= $counts['waiting'] ?></div>
</div>
<table>
<thead>
<tr>
<th style="width:36%;">Market</th>
<th style="width:24%;">Time (Open → Close)</th>
<th style="width:18%;">Status</th>
<th style="width:22%;">Result / Notes</th>
</tr>
</thead>
<tbody>
<?php foreach ($rowsForDisplay as $r): 
list($mkt, $timeLabel, $cls, $text, $res) = array_pad($r, 5, '-');
$map = ['green'=>'green','yellow'=>'yellow','red'=>'red','black'=>'black','blue'=>'blue'];
$st = isset($map[$cls]) ? $map[$cls] : 'black';
?>
<tr>
<td><strong><?= htmlspecialchars($mkt) ?></strong></td>
<td><?= htmlspecialchars($timeLabel) ?></td>
<td><span class="status <?= $st ?>"><?= htmlspecialchars($text) ?></span></td>
<td><?= ($res === '-') ? htmlspecialchars($text === '-' ? '' : $text) : htmlspecialchars($res) ?></td>
</tr>
<?php endforeach; ?>
</tbody>
</table>
</div>
</body>
</html>
<?php
$mysqli->close();
?>
